using System;
using System.Threading;
using System.Windows.Forms;
using System.Text;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.VAL;
using gov.va.med.vbecs.DAL.VistALink.Client;
using gov.va.med.vbecs.ExceptionManagement;

namespace gov.va.med.vbecs.GUI
{
	/// <summary>
	/// Base class for both VBECS and VBECS administrator applications' main forms.
	/// Implements global exception handling, app shutdown cleanup etc. 
	/// 
	/// Please be careful with this class as it affects TWO applications - both VBECS and VBECS Administrator. 
	/// 
	/// </summary>
	public class FrmAppMainFormBase : VbecsBaseForm
	{
		private static FrmAppMainFormBase _appMainForm = null;
		private static bool _appShutdownInProgress = false;
		private static bool _appShutdownCleanupStarted = false;
		private static object _appThreadSafetyLock = new object();
		private static bool _dbUnavailableFlag = false;
		private static bool _dontAskForAppExitConfirmation = false;
		private static Form _activeDialog = null;

		/// <summary>
		/// Constructor
		/// </summary>
		public FrmAppMainFormBase() : base() {}

		/// <summary>
		/// StopAllWorkerThreads
		/// </summary>
		protected virtual void StopAllWorkerThreads() {}

		/// <summary>
		/// StopAllTimers
		/// </summary>
		protected virtual void StopAllTimers() {}

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing ) // careful here! one should only reference other objects when object disposal is in progress otherwise you'll crash during finalization.
				PerformApplicationShutdownCleanup(false);       //CR3311 changed (from true) to do exit lock cleanup

			base.Dispose( disposing );
		}

		// Please, please, please: exception handling below is somewhat a global thing. It's fine to change it, but 
		// make sure 1) you know the internals of exception handling very well; 2) you don't break everything with your change;
		// 3) make sure to test both VBECS and VBECS Administrator when making a change.
		// Thanks :). 

		/// <summary>
		/// InitializeGlobalExceptionHandling
		/// </summary>
		protected static void InitializeGlobalExceptionHandling()
		{
			AttachUnhandledExceptionHandlers();
		}

		private static void AttachUnhandledExceptionHandlers()
		{		
			AppDomain.CurrentDomain.UnhandledException += new UnhandledExceptionEventHandler( OnUnhandledException ); // CLR unhandled exceptions
			Application.ThreadException += new ThreadExceptionEventHandler( OnGuiUnhandedException ); // GUI unhandled exceptions
		}

		private static void DetachUnhandledExceptionHandlers()
		{		
			AppDomain.CurrentDomain.UnhandledException -= new UnhandledExceptionEventHandler( OnUnhandledException ); 
			Application.ThreadException -= new ThreadExceptionEventHandler( OnGuiUnhandedException ); 
		}

		// CLR unhandled exception
		private static void OnUnhandledException( object sender, UnhandledExceptionEventArgs e ) 
		{
			HandleUnhandledException( e.ExceptionObject );
		}

		// Windows Forms unhandled exception
		private static void OnGuiUnhandedException( object sender, ThreadExceptionEventArgs e ) 
		{
			HandleUnhandledException( e.Exception );
		}
		
		// Please do not add cleanup code to this method, use cleanup method instead.
		private static void HandleUnhandledException( object exceptionObject )
		{
			if( exceptionObject is ThreadAbortException )	// We do not want to handle ThreadAbortException because it's a legitimate one. 
				return;										// So we'll just keep it transparent and let 'em threads die :). 

			if( IsRecoverableActiveFormException( exceptionObject ) )
				HandleRecoverableActiveFormException( exceptionObject );
			else
				HandleFatalApplicationException( exceptionObject );
		}

        // CR2761: extended number of recoverable exceptions 
        //CR2113 - adding SqlTimeoutException
		private static bool IsRecoverableActiveFormException( object exceptionObject )
		{
			return ( exceptionObject is VistALinkNetworkErrorException ) || 
                    ( exceptionObject is RowVersionException ) ||
                    (exceptionObject is SqlConnectionException) ||
                    ( exceptionObject is VistALinkClientRPCException) || 
                     ( exceptionObject is SqlTimeoutException);
		}

		private static Form SafelyGetActiveFormToBeClosedBecauseOfRecoverableException()
		{
			if( _appMainForm == null ) // Recoverable exceptions can only happen when main form is shown
				throw( new BaseApplicationException( StrRes.SysErrMsg.Common.RecoverableExceptionOccuredWithoutMainForm().ResString ) );

			Form _activeForm = _appMainForm.ActiveMdiChild; // Best bet

			if( _activeForm == _appMainForm || _activeForm == null )
				_activeForm = Form.ActiveForm; // Dialog out of MDI

			if (_activeForm == null)
				_activeForm = _activeDialog;

			//if( _activeForm == null )
			//	throw( new BaseApplicationException( StrRes.SysErrMsg.Common.RecoverableExceptionOccuredWithoutMainForm().ResString ) );

			//if( _activeForm == _appMainForm )
			//	throw( new BaseApplicationException( StrRes.SysErrMsg.Common.RecoverableExceptionOccuredWhenOnlyMainFormWasAvailable().ResString ) );

			return _activeForm;
		}

		private static void HandleRecoverableActiveFormException( object exceptionObject )
		{
            // Write exception information into the even log. 
            ExceptionManager.Publish( exceptionObject as Exception );
		    Form _activeForm = SafelyGetActiveFormToBeClosedBecauseOfRecoverableException();

            // doing it here so that message boxes below do not obstruct active form status
		    if( exceptionObject is RowVersionException )
            {
                GuiMessenger.ShowMessageBox( StrRes.SysErrMsg.Common.DataWasNotSavedBecauseOfRowversionViolation() );
                ForceActiveFormToClose( _activeForm );
                return;
            }
            // CR2761:
            // This exception happens only when RPC call fails due connection error
            if( exceptionObject is VistALinkNetworkErrorException )
            {
                GuiMessenger.ShowMessageBox( StrRes.SysErrMsg.Common.VistALinkNetworkFailure() );
                ForceActiveFormToClose( _activeForm );
                //restore connection is necessary
                if(!VistALink.EnsureAvailabilityWithoutFailureNotification())
                    GuiMessenger.ShowMessageBox( StrRes.SysErrMsg.Common.VistALinkNetworkFailure() );
                return;
            }
            if( exceptionObject is VistALinkClientRPCException )
            {
                GuiMessenger.ShowMessageBox( StrRes.SysErrMsg.VistALink.Interupted() );
                ForceActiveFormToClose( _activeForm );
                //restore connection is necessary
                if(!VistALink.EnsureAvailabilityWithoutFailureNotification())
                    GuiMessenger.ShowMessageBox( StrRes.SysErrMsg.Common.VistALinkNetworkFailure() );
                return;
            }
            if(exceptionObject is SqlConnectionException)
            {
                GuiMessenger.ShowMessageBox(StrRes.SysErrMsg.Database.DbConnectivityFailureGeneral());
                ForceActiveFormToClose(_activeForm);
                return;
            }
            if (exceptionObject is SqlTimeoutException)     //CR2113
            {
                GuiMessenger.ShowMessageBox(StrRes.SysErrMsg.Database.DbQueryTimeout());
                return;
            }            
		}

		private static void ForceActiveFormToClose( Form activeForm )
		{
            if (activeForm == null) return;
				//throw( new ArgumentNullException( "activeForm" ) );

				activeForm.Close();
		}

		private static void HandleFatalApplicationException( object exceptionObject )
		{
			try
			{
				PerformMainFormTimersCleanup();

				_dbUnavailableFlag = Utility.IsDbConnectivityError( exceptionObject );

				ReportFatalApplicationException( exceptionObject, true );
			}
			catch
			{				
				// Extreme case - exception in exception handling - catching everything
				// WARNING: this pattern is not to be used in other places in a code. 
				// It should be ONLY implemented in a top level form. And here it is. 
				try
				{
					controls.UnhandledExceptionDialog.DisplayExceptionInExceptionHandlingMessage();

					if( AppMainForm != null )
						AppMainForm.Refresh();
				}
				catch
				{
					DetachUnhandledExceptionHandlers(); // preventing repetitious errors
					// Suppressing everything there because this is the worst case 
					// that can happen - exception in unhandled exceptions handler.
				}
			}
			finally
			{
				SafelyExitApplicationAfterFatalError();
			}
		}

		private static void SafelyExitApplicationAfterFatalError()
		{
			try
			{
				DontAskForAppExitConfirmation = true;

                if (!_dbUnavailableFlag) PerformLockingCleanup();   //meh

				ExitApplication( false, true );
			}
			catch( Exception xcp )
			{
				HandleExceptionInSafeShutdown( xcp );
			}
		}

		private static void HandleExceptionInSafeShutdown( object exceptionObject )
		{
			try
			{
				DetachUnhandledExceptionHandlers(); // preventing infinite loop
				ReportFatalApplicationException( exceptionObject, false );
			}
			catch
			{
				// Extreme case - exception in exception handling - catching everything
				// WARNING: this pattern is not to be used in other places in a code. 
				// It should be ONLY implemented in a top level form. And here it is. 
			}
		}

		/// <summary>
		/// ReportFatalApplicationException
		/// </summary>
		/// <param name="exceptionObject"></param>
		/// <param name="displayErrorToUser"></param>
		public static void ReportFatalApplicationException( object exceptionObject, bool displayErrorToUser )
		{
			Exception _managedException = ExceptionManager.ConvertExceptionObjectToManagedException( exceptionObject );
			ExceptionManager.Publish( _managedException );

			if( !displayErrorToUser )
				return;

            if (_dbUnavailableFlag)
                DisplayDbConnectivityFailureErrorMessage(exceptionObject);
            else
                new controls.UnhandledExceptionDialog(_managedException).ShowDialog();
		}

		/// <summary>
		/// DisplayDbConnectivityFailureErrorMessage
		/// </summary>
		/// <param name="exceptionObject"></param>
		private static void DisplayDbConnectivityFailureErrorMessage( object exceptionObject )
		{
			GuiMessenger.ShowMessageBox( Utility.FormErrorMessageBasedOnDbConnectivityErrorObject( exceptionObject ) );
		}

		/// <summary>
		/// bypassUnlockCode - is for when the user doesn't select a division and just wants out of the app
		/// </summary>
		/// <param name="isNormalShutdown"></param>
		/// <param name="bypassUnlockCode"></param>
		protected static void ExitApplication( bool isNormalShutdown, bool bypassUnlockCode )
		{
			lock( _appThreadSafetyLock )
			{
				if( _appShutdownInProgress )
					return;

				_appShutdownInProgress = true;
				_appShutdownInProgress = true;
			}

            PerformApplicationShutdownCleanup(bypassUnlockCode);

			Environment.Exit( isNormalShutdown ? 0 : 1 );
		}

		/// <summary>
		/// PerformApplicationShutdownCleanup
		/// </summary>
		/// <param name="bypassUnlockCode"></param>
		protected static void PerformApplicationShutdownCleanup(bool bypassUnlockCode)
		{			
			lock( _appThreadSafetyLock )
			{
				if( _appShutdownCleanupStarted )
					return;

				_appShutdownCleanupStarted = true;
			}

			// Do all clean-ups below
			VistALink.Disconnect();

			if( !_dbUnavailableFlag && !bypassUnlockCode)
				PerformLockingCleanup();

			PerformMainFormTimersCleanup();
			PerformMainFormThreadsCleanup();			
		}

		/// <summary>
		/// PerformLockingCleanup
		/// </summary>
		public static void PerformLockingCleanup()
		{
			if( Common.LogonUser.LogonUserName != null)        //CR3311  - removing ThisUser
			{
				//Unlock all records -- after doing process check.. If GetVbecsSessionCount <> 1, then only remove the current session locks
				//This method is fired on Vbecs login, and on application exit
				if (Common.Utility.GetVbecsSessionCount(Common.LogonUser.LogonUserName) == 1)
				{
					//Since we're the only session open, unlock ALL records for the current user
					BOL.LockManager.UnlockAllRecordsForUser(Common.LogonUser.LogonUserName, true);
				}
				else
				{
					//There are multiple sessions open, so only unlock our current locks (this is for exit)
					BOL.LockManager.UnlockAllRecordsForUser(Common.LogonUser.LogonUserName , false);
				}
			}
		}
		
		/// <summary>
		/// PerformMainFormThreadsCleanup
		/// </summary>
		private static void PerformMainFormThreadsCleanup()
		{
			if( _appMainForm != null ) 
				_appMainForm.StopAllWorkerThreads();
		}

		/// <summary>
		/// PerformMainFormTimersCleanup
		/// </summary>
		private static void PerformMainFormTimersCleanup()
		{
			if( _appMainForm != null )
				_appMainForm.StopAllTimers();
		}

		/// <summary>
		/// Get/Set AppMainForm
		/// </summary>
		protected static FrmAppMainFormBase AppMainForm
		{
			get
			{
				return _appMainForm;
			}
			set
			{
				lock( _appThreadSafetyLock )
					_appMainForm = value;
			}
		}

		/// <summary>
		/// Get/Set DontAskForAppExitConfirmation
		/// </summary>
		protected static bool DontAskForAppExitConfirmation
		{
			get
			{
				return _dontAskForAppExitConfirmation;
			}
			set
			{
				lock( _appThreadSafetyLock )
					_dontAskForAppExitConfirmation = value;
			}
		}

		/// <summary>
		/// RegisterDialog
		/// </summary>
		/// <param name="dlg"></param>
		protected void RegisterDialog(Form dlg)
		{
			_activeDialog = dlg;
		}
	}
}
